<?php
// public/submit.php
require __DIR__ . '/../includes/db.php';
require __DIR__ . '/../includes/helpers.php';

$flow = $_POST['flow'] ?? '';

/**
 * Common helpers
 */
function save_base64_photo(string $dataUri, string $prefix): array {
  if (!preg_match('/^data:image\/(png|jpeg|jpg);base64,/', $dataUri)) {
    return [false, 'Photo invalid', null];
  }
  $cfg = require __DIR__ . '/../includes/config.php';
  $upload_dir = $cfg['upload_dir'];
  if (!is_dir($upload_dir)) mkdir($upload_dir,0755,true);

  $type = (strpos($dataUri,'image/png')!==false) ? 'png' : 'jpg';
  $bin  = base64_decode(substr($dataUri, strpos($dataUri, ',')+1));
  $fname = $prefix . '_' . time() . '_' . bin2hex(random_bytes(6)) . '.' . $type;
  $fpath = $upload_dir . DIRECTORY_SEPARATOR . $fname;
  file_put_contents($fpath, $bin);

  $photo_url = rtrim($cfg['upload_base'], '/\\') . '/' . $fname;
  return [true, null, $photo_url];
}

if ($flow === 'register') {
  // required inputs
  $date  = $_POST['date'] ?? '';
  $time  = $_POST['time'] ?? '';
  $ward  = trim($_POST['ward'] ?? '');
  $house = trim($_POST['house_no'] ?? '');
  $name  = trim($_POST['name'] ?? '');
  $phone = validate_phone($_POST['phone'] ?? '');
  $waste = $_POST['waste_segregation'] ?? '';
  $photo = $_POST['photo_data'] ?? '';

  // geo (raw) — from hidden fields populated by app.js
  $geo_lat      = $_POST['geo_lat'] ?? null;
  $geo_lng      = $_POST['geo_lng'] ?? null;
  $geo_addr     = trim($_POST['geo_addr'] ?? '');
  $geo_village  = trim($_POST['geo_village'] ?? '');
  $geo_district = trim($_POST['geo_district'] ?? '');
  $geo_state    = trim($_POST['geo_state'] ?? '');
  $geo_country  = trim($_POST['geo_country'] ?? '');
  $geo_cc       = strtoupper(trim($_POST['geo_cc'] ?? ''));
  $captured_at  = $_POST['captured_at'] ?? null;

  // validation
  $errors = [];
  if (!$date || !$time) $errors[]='Date/time required';
  if (!$ward) $errors[]='Ward required';
  if (!$house) $errors[]='House no required';
  if (!$name) $errors[]='Name required';
  if (!$phone) $errors[]='Phone invalid';
  if (!$waste) $errors[]='Waste segregation required';
  if (!$photo) $errors[]='Photo required';
  if ($errors) json_response(['success'=>false,'message'=>implode(', ', $errors)]);

  // enforce unique phone
  $st = $pdo->prepare('SELECT id FROM residents WHERE phone=:p LIMIT 1');
  $st->execute(['p'=>$phone]);
  if ($st->fetch()) json_response(['success'=>false,'message'=>'Phone already registered']);

  // save photo
  [$ok, $perr, $photo_path] = save_base64_photo($photo, 'reg');
  if (!$ok) json_response(['success'=>false,'message'=>$perr ?? 'Photo invalid']);

  // timestamp
  $created_at = date('Y-m-d H:i:s', strtotime($date.' '.$time.':00'));

  // insert resident + training #1 (initial)
  $pdo->beginTransaction();
  try {
    // residents (now stores geo columns for registration capture)
    $ins = $pdo->prepare('INSERT INTO residents
      (ward, house_no, name, phone, waste_segregation, photo_path,
       created_at, updated_at, trainings_completed,
       reg_lat, reg_lng, reg_address, reg_village, reg_district, reg_state,
       reg_country, reg_country_code, reg_captured_at)
      VALUES
      (:w,:h,:n,:p,:ws,:ph,:c,:c,1,
       :lat,:lng,:addr,:village,:district,:state,:country,:cc,:captured)');
    $ins->execute([
      'w'=>$ward,
      'h'=>$house,
      'n'=>$name,
      'p'=>$phone,
      'ws'=>$waste,
      'ph'=>$photo_path,
      'c'=>$created_at,
      'lat'=>($geo_lat !== null && $geo_lat !== '') ? $geo_lat : null,
      'lng'=>($geo_lng !== null && $geo_lng !== '') ? $geo_lng : null,
      'addr'=>$geo_addr ?: null,
      'village'=>$geo_village ?: null,
      'district'=>$geo_district ?: null,
      'state'=>$geo_state ?: null,
      'country'=>$geo_country ?: null,
      'cc'=>$geo_cc ?: null,
      'captured'=>$captured_at ?: null,
    ]);
    $resident_id = (int)$pdo->lastInsertId();

    // trainings #1 (store same geo snapshot too)
    $insT = $pdo->prepare('INSERT INTO trainings
      (resident_id, training_number, waste_segregation, photo_path,
       training_date, created_at,
       lat, lng, address, village, district, state, country, country_code, captured_at)
      VALUES
      (:rid,1,:ws,:ph,:td,:now,
       :lat,:lng,:addr,:village,:district,:state,:country,:cc,:captured)');
    $insT->execute([
      'rid'=>$resident_id,
      'ws'=>$waste,
      'ph'=>$photo_path,
      'td'=>$created_at,
      'now'=>now(),
      'lat'=>($geo_lat !== null && $geo_lat !== '') ? $geo_lat : null,
      'lng'=>($geo_lng !== null && $geo_lng !== '') ? $geo_lng : null,
      'addr'=>$geo_addr ?: null,
      'village'=>$geo_village ?: null,
      'district'=>$geo_district ?: null,
      'state'=>$geo_state ?: null,
      'country'=>$geo_country ?: null,
      'cc'=>$geo_cc ?: null,
      'captured'=>$captured_at ?: null,
    ]);

    $pdo->commit();
    json_response(['success'=>true,'message'=>'Registration saved (Training First recorded).']);
  } catch (Exception $e) {
    $pdo->rollBack();
    json_response(['success'=>false,'message'=>'DB error: '.$e->getMessage()]);
  }
}

if ($flow === 'officer') {
  $resident_id = (int)($_POST['resident_id'] ?? 0);
  $waste = $_POST['waste_segregation'] ?? '';
  $date  = $_POST['date'] ?? '';
  $time  = $_POST['time'] ?? '';
  $photo = $_POST['photo_data'] ?? '';

  // geo from either _off or generic names
  $geo_lat      = $_POST['geo_lat_off'] ?? $_POST['geo_lat'] ?? null;
  $geo_lng      = $_POST['geo_lng_off'] ?? $_POST['geo_lng'] ?? null;
  $geo_addr     = trim($_POST['geo_addr_off'] ?? $_POST['geo_addr'] ?? '');
  $geo_village  = trim($_POST['geo_village_off'] ?? $_POST['geo_village'] ?? '');
  $geo_district = trim($_POST['geo_district_off'] ?? $_POST['geo_district'] ?? '');
  $geo_state    = trim($_POST['geo_state_off'] ?? $_POST['geo_state'] ?? '');
  $geo_country  = trim($_POST['geo_country_off'] ?? $_POST['geo_country'] ?? '');
  $geo_cc       = strtoupper(trim($_POST['geo_cc_off'] ?? $_POST['geo_cc'] ?? ''));
  $captured_at  = $_POST['captured_at_off'] ?? $_POST['captured_at'] ?? null;

  $errors = [];
  if (!$resident_id) $errors[]='Resident missing';
  if (!$waste) $errors[]='Waste segregation required';
  if (!$date || !$time) $errors[]='Date/time required';
  if (!$photo) $errors[]='Photo required';
  if ($errors) json_response(['success'=>false,'message'=>implode(', ', $errors)]);

  // verify resident + trainings_completed
  $st = $pdo->prepare('SELECT id, trainings_completed FROM residents WHERE id=:id LIMIT 1');
  $st->execute(['id'=>$resident_id]);
  $res = $st->fetch();
  if (!$res) json_response(['success'=>false,'message'=>'Resident not found']);

  $trainings_done = (int)$res['trainings_completed']; // after registration this is 1
  if ($trainings_done >= 3) json_response(['success'=>false,'message'=>'All trainings completed']);
  $next_num = $trainings_done + 1;  // 2 then 3

  // save photo
  [$ok, $perr, $photo_path] = save_base64_photo($photo, 't'.$next_num.'_'.$resident_id);
  if (!$ok) json_response(['success'=>false,'message'=>$perr ?? 'Photo invalid']);

  $training_dt = date('Y-m-d H:i:s', strtotime($date.' '.$time.':00'));

  $pdo->beginTransaction();
  try {
    // insert next training with geo
    $insT = $pdo->prepare('INSERT INTO trainings
      (resident_id, training_number, waste_segregation, photo_path,
       training_date, created_at,
       lat, lng, address, village, district, state, country, country_code, captured_at)
      VALUES
      (:rid,:tn,:ws,:ph,:td,:now,
       :lat,:lng,:addr,:village,:district,:state,:country,:cc,:captured)');
    $insT->execute([
      'rid'=>$resident_id,
      'tn'=>$next_num,
      'ws'=>$waste,
      'ph'=>$photo_path,
      'td'=>$training_dt,
      'now'=>now(),
      'lat'=>($geo_lat !== null && $geo_lat !== '') ? $geo_lat : null,
      'lng'=>($geo_lng !== null && $geo_lng !== '') ? $geo_lng : null,
      'addr'=>$geo_addr ?: null,
      'village'=>$geo_village ?: null,
      'district'=>$geo_district ?: null,
      'state'=>$geo_state ?: null,
      'country'=>$geo_country ?: null,
      'cc'=>$geo_cc ?: null,
      'captured'=>$captured_at ?: null,
    ]);

    // increment trainings_completed
    $upd = $pdo->prepare('UPDATE residents
      SET trainings_completed = trainings_completed + 1, updated_at = :u
      WHERE id = :id');
    $upd->execute(['u'=>now(),'id'=>$resident_id]);

    $pdo->commit();
    json_response(['success'=>true,'message'=>'Training saved successfully']);
  } catch (Exception $e) {
    $pdo->rollBack();
    json_response(['success'=>false,'message'=>'DB error: '.$e->getMessage()]);
  }
}

json_response(['success'=>false,'message'=>'Invalid flow']);
